//===-- ThreadPlanShouldStopHere.h ------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//

#ifndef liblldb_ThreadPlanShouldStopHere_h_
#define liblldb_ThreadPlanShouldStopHere_h_

// C Includes
// C++ Includes
// Other libraries and framework includes
// Project includes
#include "lldb/Target/ThreadPlan.h"

namespace lldb_private {

// This is an interface that ThreadPlans can adopt to allow flexible modifications of the behavior
// when a thread plan comes to a place where it would ordinarily stop.  If such modification makes
// sense for your plan, inherit from this class, and when you would be about to stop (in your ShouldStop
// method), call InvokeShouldStopHereCallback, and if that returns a non-NULL plan, execute that
// plan instead of stopping.
//
// The classic example of the use of this is ThreadPlanStepInRange not stopping in frames that have
// no debug information.
//
// This class also defines a set of flags to control general aspects of this "ShouldStop" behavior.
// A class implementing this protocol needs to define a default set of flags, and can provide access to
// changing that default flag set if it wishes.

class ThreadPlanShouldStopHere
{
public:
    enum
    {
        eNone = 0,
        eAvoidInlines = (1 << 0),
        eAvoidNoDebug = (1 << 1)
    };

    //------------------------------------------------------------------
    // Constructors and Destructors
    //------------------------------------------------------------------
    ThreadPlanShouldStopHere (ThreadPlan *owner,
                              ThreadPlanShouldStopHereCallback callback = NULL,
                              void *baton = NULL);
    virtual
    ~ThreadPlanShouldStopHere();

    void
    SetShouldStopHereCallback (ThreadPlanShouldStopHereCallback callback, void *baton);

    ThreadPlan *
    InvokeShouldStopHereCallback ();

    lldb_private::Flags &
    GetFlags ()
    {
        return m_flags;
    }

    const lldb_private::Flags &
    GetFlags () const
    {
        return m_flags;
    }

protected:
    // Implement this, and call it in the plan's constructor to set the default flags.
    virtual void SetFlagsToDefault () = 0;

    //------------------------------------------------------------------
    // Classes that inherit from ThreadPlanShouldStopHere can see and modify these
    //------------------------------------------------------------------
    ThreadPlanShouldStopHereCallback m_callback;
    void * m_baton;
    ThreadPlan *m_owner;
    lldb_private::Flags m_flags;

private:
    //------------------------------------------------------------------
    // For ThreadPlanShouldStopHere only
    //------------------------------------------------------------------

    DISALLOW_COPY_AND_ASSIGN (ThreadPlanShouldStopHere);

};

} // namespace lldb_private

#endif  // liblldb_ThreadPlanShouldStopHere_h_
