#!/bin/bash

if ! test "${1}" = "d"; then
    exec dbus-run-session -- ${0} d
fi

set -e -u

LOG=run-tests.log
QUEUE=Save_As_PDF
FRONTEND=cpdb-text-frontend
BACKEND=./file
BACKEND_INFO_SOURCE_DIR=../data
BACKEND_INFO=org.openprinting.Backend.FILE
FILE_TO_PRINT=testpage.pdf
OUTPUT_FILE=test.pdf
FRONTEND_PID=
BACKEND_PID=

cleanup() {
    # Give the sub-processes time to die a natural death ...
    sleep 1;
    # Kill frontend
    if test -n "${FRONTEND_PID}"; then
	echo kill -KILL ${FRONTEND_PID}
    fi
    # Kill backend
    if test -n "${BACKEND_PID}"; then
	echo kill -KILL ${BACKEND_PID}
    fi
    # Remove the copy of the backend info file
    rm -f $CPDB_BACKEND_INFO_DIR/$BACKEND_INFO
    # Remove "printed" test page
    rm -f $OUTPUT_FILE
    # Show log
    cat $LOG
    # Remove the log file
    rm -f $LOG
}

trap cleanup 0 EXIT INT QUIT ABRT PIPE TERM

# Create the log file
rm -f $LOG
touch $LOG

# Start backend
$BACKEND &
BACKEND_PID=$!

# Set backend info directory and copy the info file to there
# In its original place, in the data/ sub-directory it is together
# with the systemd service file which gets confused as a backend info
# file by cpdb-libs, therefore we copy it.
export CPDB_BACKEND_INFO_DIR=.
cp $BACKEND_INFO_SOURCE_DIR/$BACKEND_INFO $CPDB_BACKEND_INFO_DIR

# Run the test frontend with a session D-Bus and feed in commands.
( \
  sleep 3; \
  echo get-all-options $QUEUE FILE; \
  echo print-file $FILE_TO_PRINT $QUEUE FILE; \
  echo $OUTPUT_FILE; \
  sleep 3; \
  echo stop \
) | $FRONTEND > $LOG 2>&1 & FRONTEND_PID=$!

# Give the frontend a maximum of 10 seconds to run and then kill it, to avoid
# the script getting stuck if stopping it fails.
i=0
while kill -0 $FRONTEND_PID >/dev/null 2>&1; do
    i=$((i+1))
    if test $i -ge 10; then
	kill -KILL $FRONTEND_PID >/dev/null 2>&1 || true
	FRONTEND_PID=
	echo "FAIL: Frontend keeps running!"
	exit 1
    fi
    sleep 1
done
FRONTEND_PID=

# Stop the backend
if kill -0 $BACKEND_PID >/dev/null 2>&1; then
    kill -TERM $BACKEND_PID
    # Give the backend a maximum of 3 seconds to shut down and then kill it,
    # to avoid the script getting stuck if shutdown fails.
    i=0
    while kill -0 $BACKEND_PID >/dev/null 2>&1; do
	i=$((i+1))
	if test $i -ge 3; then
	    kill -KILL $BACKEND_PID >/dev/null 2>&1 || true
	    BACKEND_PID=
	    echo "FAIL: Backend did not shut down!"
	    exit 1
	fi
	sleep 1
    done
fi
BACKEND_PID=

echo

# Does the printer appear in the initial list of available printers?
echo "Initial listing of the printer:"
( grep '^Printer '$QUEUE'$' $LOG ) || \
    ( echo "Printer $QUEUE not listed!"; \
      exit 1 )

echo

# Is the list of options empty?
echo "Option list empty:"
( grep 'Retrieved 0 options' $LOG ) || \
    ( echo "Backend has options!"; \
      exit 1 )

echo

# Did the successful submission of a print job get confirmed?
echo "Print file actually written and identical with input file:"
if [ ! -r $OUTPUT_FILE ]; then
    echo "Print file not written!"
    exit 1
else
    ls -l $OUTPUT_FILE
fi

# Did the input file make it into the output file?
if ! diff -q "$FILE_TO_PRINT" "$OUTPUT_FILE"; then
    echo "Print file and input file differ!"
    exit 1
else
    echo "File correctly printed!"
fi

echo

if grep -q "Stopping front end" $LOG; then
    echo "Frontend correctly shut down"
else
    echo "FAIL: Frontend not correctly stopped!"
    exit 1
fi

echo "SUCCESS!"

exit 0
